<?php
/*
 * Copyright © 2018-2025 RBSoft (Ravi Patel). All rights reserved.
 *
 * Author: Ravi Patel
 * Website: https://rbsoft.org/downloads/sms-gateway
 *
 * This software is licensed, not sold. Buyers are granted a limited, non-transferable license
 * to use this software exclusively on a single domain, subdomain, or computer. Usage on
 * multiple domains, subdomains, or computers requires the purchase of additional licenses.
 *
 * Redistribution, resale, sublicensing, or sharing of the source code, in whole or in part,
 * is strictly prohibited. Modification (except for personal use by the licensee), reverse engineering,
 * or creating derivative works based on this software is strictly prohibited.
 *
 * Unauthorized use, reproduction, or distribution of this software may result in severe civil
 * and criminal penalties and will be prosecuted to the fullest extent of the law.
 *
 * For licensing inquiries or support, please visit https://support.rbsoft.org.
 */

namespace App\Imports;

use App\Enums\CampaignStatus;
use App\Models\Campaign;
use App\Monitor;
use App\Rules\MobileNumber;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Concerns\Importable;

class MessagesImport extends TrackedImport
{
    use Importable;

    private Campaign $campaign;

    public function __construct(Campaign $campaign)
    {
        $this->campaign = $campaign;
    }

    public function chunkSize(): int
    {
        return 1000;
    }

    protected function monitor(): Monitor
    {
        return Monitor::for(Campaign::class, $this->campaign->id);
    }

    /**
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function processRow(array $data): void
    {
        Validator::make(
            $data,
            ['mobile_number' => ['required', new MobileNumber()]]
        )->validate();

        $allowed =
            $this->campaign->options->get('force', false) ||
            $this->campaign->user->blacklist()
                                 ->whereMobileNumber($data['mobile_number'])
                                 ->doesntExist();

        if ($allowed) {
            $succeeded = $this->monitor()->processed - $this->monitor()->failures;
            $senderCount = $this->campaign->senders->count();
            $senderIndex = $senderCount > 0 ? $succeeded % $senderCount : 0;
            $this->campaign->addMessage(
                $data['mobile_number'],
                $this->campaign->senders->get($senderIndex),
                Arr::except($data, 'mobile_number')
            );
        }
    }

    protected function beforeImport(): void
    {
        $this->campaign->update(['status' => CampaignStatus::Processing]);
    }

    protected function afterImport(): void
    {
        $this->campaign->update(['status' => CampaignStatus::Processed]);
    }
}
