<?php
/*
 * Copyright © 2018-2025 RBSoft (Ravi Patel). All rights reserved.
 *
 * Author: Ravi Patel
 * Website: https://rbsoft.org/downloads/sms-gateway
 *
 * This software is licensed, not sold. Buyers are granted a limited, non-transferable license
 * to use this software exclusively on a single domain, subdomain, or computer. Usage on
 * multiple domains, subdomains, or computers requires the purchase of additional licenses.
 *
 * Redistribution, resale, sublicensing, or sharing of the source code, in whole or in part,
 * is strictly prohibited. Modification (except for personal use by the licensee), reverse engineering,
 * or creating derivative works based on this software is strictly prohibited.
 *
 * Unauthorized use, reproduction, or distribution of this software may result in severe civil
 * and criminal penalties and will be prosecuted to the fullest extent of the law.
 *
 * For licensing inquiries or support, please visit https://support.rbsoft.org.
 */

namespace App\Http\Controllers;

use App\Data\Settings\AutoRetrySettingsData;
use App\Data\Settings\CreditsSettingsData;
use App\Data\Settings\EmailToMessageSettingsData;
use App\Data\Settings\FeaturesSettingsData;
use App\Data\Settings\PromptsSettingsData;
use App\Data\Settings\QrCodeSettingsData;
use App\Data\Settings\TrialSettingsData;
use App\Data\Settings\WebhookServerSettingsData;
use App\Models\Plan;
use App\Models\Setting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Redirect;
use Inertia\Inertia;
use Inertia\Response;

class MiscellaneousSettingsController extends Controller
{
    /**
     * Show the form for editing the specified resource.
     *
     * @throws \Illuminate\Auth\Access\AuthorizationException
     */
    public function edit(): Response
    {
        Gate::authorize('updateAny', Setting::class);

        $webhookServer = Arr::only(
            config('webhook-server'),
            ['tries', 'timeout_in_seconds', 'verify_ssl']
        );

        $extra = [
            'queue_up' => config('queue.default') === 'database',
            'imap' => config('imap.accounts.default', []),
            'webhook_server' => $webhookServer,
            'firebase' => [
                'project_id' => config('firebase.projects.app.credentials.project_id')
            ]
        ];

        return Inertia::render('Settings/Misc/Show', [
            'plans' => Plan::all(['id', 'name', 'price', 'currency'])->append('label'),
            'settings' => config('misc') + $extra,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function webhook(WebhookServerSettingsData $data): RedirectResponse
    {
        Setting::store('webhook-server.tries', $data->tries);
        Setting::store('webhook-server.timeout_in_seconds', $data->timeout);
        Setting::store('webhook-server.verify_ssl', $data->verifySsl);

        return Redirect::back();
    }

    public function credits(CreditsSettingsData $data): RedirectResponse
    {
        Setting::store('misc.credits.received.amount', $data->received);
        Setting::store('misc.credits.sms.amount', $data->sms);
        Setting::store('misc.credits.sms.per_part', $data->perPart);
        Setting::store('misc.credits.mms.amount', $data->mms);
        Setting::store('misc.credits.ussd_pull.amount', $data->ussdPull);
        Setting::store('misc.credits.call.amount', $data->call);
        Setting::store('misc.credits.webhook_call.amount', $data->webhookCall);
        Setting::store('misc.credits.message_to_email.amount', $data->messageToEmail);
        Setting::store('misc.credits.email_to_message.amount', $data->emailToMessage);

        return Redirect::back();
    }

    public function features(FeaturesSettingsData $data): RedirectResponse
    {
        if ($data->serviceAccountJson) {
            Setting::store('firebase.projects.app.credentials', json_decode($data->serviceAccountJson->getContent()));
        }
        Setting::store('queue.default', $data->queueUp ? 'database' : 'sync');
        Setting::store('misc.features.show_homepage', $data->showHomepage);
        Setting::store('misc.features.realtime_dashboard', $data->realtimeDashboard);
        Setting::store('misc.features.message_to_email', $data->messageToEmail);

        return Redirect::back();
    }

    public function qrCode(QrCodeSettingsData $data)
    {
        Setting::store('misc.qr_code.lifespan', $data->lifespan);

        return Redirect::back();
    }

    public function trial(TrialSettingsData $data): RedirectResponse
    {
        Setting::store('misc.trial.plan_id', $data->plan);
        Setting::store('misc.trial.days', $data->days);
        Setting::store('misc.trial.footer', $data->footer);

        return Redirect::back();
    }

    public function prompts(PromptsSettingsData $data): RedirectResponse
    {
        Setting::store('misc.prompts.blacklist', $data->blacklist);
        Setting::store('misc.prompts.whitelist', $data->whitelist);
        Setting::store('misc.prompts.subscribe', $data->subscribe);
        Setting::store('misc.prompts.unsubscribe', $data->unsubscribe);
        Setting::store('misc.prompts.notify', $data->notify);

        return Redirect::back();
    }

    public function autoRetry(AutoRetrySettingsData $data): RedirectResponse
    {
        Setting::store('misc.features.auto_retry', $data->enabled);
        Setting::store('misc.auto_retry.max', $data->max);
        Setting::store('misc.auto_retry.change_after', $data->changeAfter);

        return Redirect::back();
    }

    public function emailToMessage(EmailToMessageSettingsData $data): RedirectResponse
    {
        Setting::store('misc.features.email_to_message', $data->enabled);
        Setting::store('misc.email_to_message.email', $data->email);
        Setting::store('imap.accounts.default.host', $data->host);
        Setting::store('imap.accounts.default.port', $data->port);
        Setting::store('imap.accounts.default.protocol', $data->protocol);
        Setting::store('imap.accounts.default.encryption', $data->encryption);
        Setting::store('imap.accounts.default.validate_cert', $data->validateCert);
        Setting::store('imap.accounts.default.username', $data->username);
        Setting::store('imap.accounts.default.password', $data->password);

        return Redirect::back();
    }
}
