<?php
/*
 * Copyright © 2018-2025 RBSoft (Ravi Patel). All rights reserved.
 *
 * Author: Ravi Patel
 * Website: https://rbsoft.org/downloads/sms-gateway
 *
 * This software is licensed, not sold. Buyers are granted a limited, non-transferable license
 * to use this software exclusively on a single domain, subdomain, or computer. Usage on
 * multiple domains, subdomains, or computers requires the purchase of additional licenses.
 *
 * Redistribution, resale, sublicensing, or sharing of the source code, in whole or in part,
 * is strictly prohibited. Modification (except for personal use by the licensee), reverse engineering,
 * or creating derivative works based on this software is strictly prohibited.
 *
 * Unauthorized use, reproduction, or distribution of this software may result in severe civil
 * and criminal penalties and will be prosecuted to the fullest extent of the law.
 *
 * For licensing inquiries or support, please visit https://support.rbsoft.org.
 */

namespace App\Http\Controllers;

use App\Data\Settings\AutoRetryUserSettingsData;
use App\Data\Settings\DashboardUserSettingsData;
use App\Data\Settings\FeaturesUserSettingsData;
use App\Data\Settings\PromptsUserSettingsData;
use App\Data\Settings\QrCodeSettingsData;
use App\Models\Setting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Redirect;
use Inertia\Inertia;
use Inertia\Response;

class UserSettingsController extends Controller
{
    public function edit(): Response
    {
        $globalSettings = Arr::only(
            config('misc'),
            ['auto_retry', 'prompts', 'features', 'qr_code', 'email_to_message', 'dashboard']
        );

        $defaultSettings = [
            'features' => [
                'auto_retry' => false,
                'email_to_message' => false,
                'message_to_email' => false,
            ]
        ];

        $userSettings = Setting::retrieve('misc', Auth::id(), []);

        return Inertia::render('Settings/User/Show', [
            'settings' => array_replace_recursive($globalSettings, $defaultSettings, $userSettings),
            'global' => $globalSettings
        ]);
    }

    public function autoRetry(AutoRetryUserSettingsData $data): RedirectResponse
    {
        Setting::store('misc.features.auto_retry', $data->enabled, Auth::id());
        Setting::store('misc.auto_retry.max', $data->max, Auth::id());
        Setting::store('misc.auto_retry.change_after', $data->changeAfter, Auth::id());

        return Redirect::back();
    }

    public function features(FeaturesUserSettingsData $data): RedirectResponse
    {
        Setting::store('misc.features.email_to_message', $data->emailToMessage, Auth::id());
        Setting::store('misc.features.message_to_email', $data->messageToEmail, Auth::id());

        return Redirect::back();
    }

    public function qrCode(QrCodeSettingsData $data)
    {
        Setting::store('misc.qr_code.lifespan', $data->lifespan, Auth::id());

        return Redirect::back();
    }

    public function prompts(PromptsUserSettingsData $data): RedirectResponse
    {
        Setting::store('misc.prompts.blacklist', $data->blacklist, Auth::id());
        Setting::store('misc.prompts.whitelist', $data->whitelist, Auth::id());
        Setting::store('misc.prompts.subscribe', $data->subscribe, Auth::id());
        Setting::store('misc.prompts.unsubscribe', $data->unsubscribe, Auth::id());
        Setting::store('misc.prompts.notify', $data->notify, Auth::id());

        return Redirect::back();
    }

    // Add dashboard update for user
    public function dashboard(DashboardUserSettingsData $data): RedirectResponse
    {
        Setting::store('misc.dashboard.campaigns', $data->campaigns, Auth::id());
        Setting::store('misc.dashboard.calls', $data->calls, Auth::id());
        Setting::store('misc.dashboard.messages', $data->messages, Auth::id());
        Setting::store('misc.dashboard.ussd_pulls', $data->ussdPulls, Auth::id());

        return Redirect::back();
    }
}
